#[cfg(feature = "ml-kem-512")]
use fips203::ml_kem_512;
use fips203::traits::{Decaps, Encaps, KeyGen, SerDes};
use hex_literal::hex;
use rand_core::SeedableRng;

#[test]
#[cfg(feature = "ml-kem-512")]
fn wasm_match() {
    let seed = 123; // Note: this should match the value giving in the web form
    let mut rng = rand_chacha::ChaCha8Rng::seed_from_u64(seed);

    // Alice runs KeyGen, and serializes ek for Bob (to bytes)
    let (alice_ek, alice_dk) = ml_kem_512::KG::try_keygen_with_rng(&mut rng).unwrap();
    let alice_ek_bytes = alice_ek.into_bytes();

    // Alice sends ek bytes to Bob
    let bob_ek_bytes = alice_ek_bytes;

    // Bob deserializes ek bytes, runs Encaps, to get ssk and serializes ct for Alice (to bytes)
    let bob_ek = ml_kem_512::EncapsKey::try_from_bytes(bob_ek_bytes).unwrap();
    let (bob_ssk, bob_ct) = bob_ek.try_encaps_with_rng(&mut rng).unwrap();
    let bob_ct_bytes = bob_ct.into_bytes();

    // Bob sends ct bytes to Alice
    let alice_ct_bytes = bob_ct_bytes;

    // Alice deserializes runs Decaps
    let alice_ct = ml_kem_512::CipherText::try_from_bytes(alice_ct_bytes).unwrap();
    let alice_ssk = alice_dk.try_decaps(&alice_ct).unwrap();

    // Alice and Bob now have the same shared secret key
    assert_eq!(bob_ssk, alice_ssk);

    // Double check the correspondence of the serialized keypair (we already have alice_ek)
    let alice_dk_bytes = alice_dk.into_bytes();
    assert!(ml_kem_512::KG::validate_keypair_with_rng_vartime(
        &mut rng,
        &alice_ek_bytes,
        &alice_dk_bytes
    ));

    assert_eq!(bob_ek_bytes, hex!("8825a351c035c89a63cd381a0026bb382257d8a40f64d28b48a69e4595847ea92bebbcb51cd0bf87b902e7ea0c2dd02f2fb665b95437cb395e2b207d4b22293bab1f43035468493306f69d3bda9d5b2a4ce9893a07d9621eb63067c90a9cbb25b9a4a8b4d060c65520a1eb9948f56a960950e2295c0253464a15064b721da86c2653c7947ddac497847a16288dfce130d5738c569c08595cb822428324367feab5a7e7c69be835b1f67457711c73364665306a08511bba94c0053e6114f1598c37fa94607970bb08995ac0acdfc58dce27573ab064d9e85547f76c98f3b3d5e58dc1c9447ee24dd3c6c3530803758787e3c79bd59933c2574f99078438b8a0521cb0942c2d45b056b160735eea6baa6a4bdb54975e8bad9c8019c3874365e294d95b07aaa1ae017b9f4955b6add08947ec73501a632f42bdab102a0ed6816e4724de3570cea5c2e83505af26687ccb69423c321bcb58dceb6cf31ab1eb9388ee952f41350f70a2956355238d7c18a24580bad18f4a1b818bb67e9e18b50a35146e9453e495b8d7890adadab7e315cfb58c92c33384b7fc3d1366a7fdcccfc59b1658152e20542530388173269c72cc8c03e8419a579e6ae1c866890c576bc256d31da82a7cf0a4823cb900cbd893c857bde06231f073b1a702105d73c3c2e3a4b01a1a2d036c32e01b94b0831b0b06fee78f1b5083bf547c9713c3ea44bf7b49743c005167d0c81b7c7b4fab2288f6176f3947aa2b3360f160d113a553740364fa60a4151a297694763b65835042b6ebb18b7a53dc7273aef26df25b746c78ad0177718b5b8dfd92a890365e452b376ea28c05a98dc5ab3ff6d21d8bd2b46536680682592cc3093615a17f5928a427737531561da02db161034060b352e1b87cf53b0035090f8217556720e1f450e4bcab94164ac03317a663b96f768b5098b7f63ab722d22285088f7dab4a8c36b2a3627226116774ab7f36fc4c050b8551a8001d993bf5811c26286eeb1b3a2775638a66b48ad963154764031765b4fa7ebe5c9f4987a5f542b1df211f38d642187b4e16c440f20699c4f6bfaffa487955bb19a1b263bbf85a9e3456f4de72d0933ebe52aef77f2812a5809e8019a3a085ba"), "ek does not match");
    assert_eq!(bob_ct_bytes, hex!("a93e7de0fe8375f09e3ab79afcb4637be7b7f5a8d065f3048fc8123fd8a1fd1a7ea71b6bafd3bcee51dc92c7bd39e8a9cc96e44620c114556e8a26d1ab9d9faa74dba5d87ab20a11bcfda1af350de23cd3d6b804a82182688b041913bde4c5ba472a83e823bb14c49ee48a08e01d4ff76f1cf8f652ee97516af3fdcf5bab7afcd0d634bb1660afaacb2b4bc7d21811a3eff8896b5a30f11d3a5a72ecd253109803068c441aedd1110c6a9f4e199c382890cdc55404e3291d385d37d332c4c0e3637329ec9b4787ae815c09cbef20f12a43b7368883b896987780fc49fe0427621ade89bf117bf8b3ad79366905d1ed5111e98d9edd8ab7c68dfd5bd26e788036d0d060c3666d3efd712651ee73149b4425b7aa1ce3b3f789f3bcfc9cd0859571a5d88064fb69cd1128211ab9839cedf9a051e961cec47a4fbcfceab7ba41dec76af6eabc0d691996751eec855e22c6917b953450b50722e51f6490347aac603c4d068e78d881803f9d992c03696fe7c7ed5b0a5902df6a7e2291f1e4ce333bae5347d258e435d5638ad11060667071e18312e3493494a4046139701230832e4a3f4b7c132b935ace9066441e6632096e2b4d5b6ad88f7601877f30357b56a63ae6877c5051a429aaa70829ad4171acc6930b2da9b23e8e0c0df7fa6bd7cf81f9611f0b5716b92ec63e513599d0098bdc3f83e3b2d3ca4e593644c952d05a771b91b93b1682a6f39846527ab71b9b9d9b4b7ef61d075eec4f1554d689912ce894447eb45db51fd483f9c8cebcd7b313b4a4ee1944935c678cf80745755bc802a1056bdd34002e47decbdd9f31ecb99760ac9ea602bbb3b10af68214941c17f7ad620e044388451c9ea451f5a1929681a61a54b15c208483e2bee1759f27c65dd1450b0ab49dc15a5d15aab5018b2a2c1e836f997f0b7a42bea76287d31274445312a610c89359a0d3e6c2b7c4ad30df6c437a256297c2de844db0229286215ba7091c1f64fd55c7bb78ce292def21758dcf5378706968291f6eb1f67931702a331824e7240c063e35645504fa18352e4bfc52469972b71291194e77b7dae27ac4"), "ct does not match");
    assert_eq!(alice_dk_bytes, hex!("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"), "dk does not match");
    assert_eq!(
        alice_ssk.into_bytes(),
        hex!("3a725a4fb17c323cf20a1d9fa8584f3f52b3ff977fdc0e7acceed32b26055797"),
        "ssk does not match"
    );
}
